<?php

namespace App\Shared\EntityFactory;

use App\News\Request\NewsCreateRequest;
use App\News\Request\NewsFullUpdateRequest;
use App\News\Request\NewsPartUpdateRequest;
use App\Shared\Entity\News;
use App\Shared\Error\NotFoundError;
use App\Shared\Repository\FileRepository;
use App\Shared\Repository\NewsCategoriesRepository;
use App\Shared\Repository\NewsRepository;
use App\Shared\Repository\NewsTypeRepository;
use Ramsey\Collection\Collection;
use Ramsey\Uuid\Uuid;

class NewsEntityFactory
{
    /** @var Collection<callable> */
    private readonly Collection $newsSetters;

    public function __construct(
        private readonly NewsRepository $newsRepository,
        private readonly NewsTypeRepository $newsTypeRepository,
        private readonly NewsCategoriesRepository $categoriesRepository,
        private readonly FileRepository $fileRepository,
    ) {
        $this->newsSetters = $this->setNewsSetters();
    }

    public function create(NewsCreateRequest $request): News
    {
        $news = new News();
        $news->setId(Uuid::uuid4()->toString());

        foreach ($request as $key => $value) {
            if ($value !== null) {
                $this->newsSetters[$key]($value, $news);
            }
        }

        return $news;
    }

    public function softUpdate(NewsFullUpdateRequest $request): News
    {
        $news = $this->newsRepository->find($request->id) ?? new News();

        foreach ($request as $key => $value) {
            if ($value !== null) {
                $this->newsSetters[$key]($value, $news);
            }
        }

        return $news;
    }

    public function hardUpdate(NewsPartUpdateRequest $request): News
    {
        $news = $this->newsRepository->find($request->id);

        if ($news === null) {
            throw new NotFoundError('News entity for update not found');
        }

        foreach ($request as $key => $value) {
            if ($value !== null) {
                $this->newsSetters[$key]($value, $news);
            }
        }

        return $news;
    }

    /** @return Collection<callable> */
    private function setNewsSetters(): Collection
    {
        $newsSetters = new Collection('callable');

        $newsSetters['id'] = function (string $id, News $news) {
            $news->setId($id);
        };
        $newsSetters['name'] = function (string $value, News $news) {
            $news->setName($value);
        };
        $newsSetters['code'] = function (string $value, News $news) {
            $news->setCode($value);
        };
        $newsSetters['active'] = function (bool $value, News $news) {
            $news->setActive($value);
        };
        $newsSetters['main_page_render'] = function (bool $value, News $news) {
            $news->setMainPageRender($value);
        };
        $newsSetters['preview_text'] = function (string $value, News $news) {
            $news->setPreviewText($value);
        };
        $newsSetters['detail_text'] = function (string $value, News $news) {
            $news->setDetailText($value);
        };
        $newsSetters['type_id'] = function (string $id, News $news) {
            $news->setType($this->newsTypeRepository->find($id));
        };
        $newsSetters['categories_id'] = function (Collection $ids, News $news) {
            foreach ($ids as $id) {
                $news->addCategory(
                    $this->categoriesRepository->find($id)
                );
            }
        };
        $newsSetters['detail_image'] = function (string $id, News $news) {
            $news->setDetailImage($this->fileRepository->find($id));
        };
        $newsSetters['preview_image'] = function (string $id, News $news) {
            $news->setPreviewImage($this->fileRepository->find($id));
        };

        return $newsSetters;
    }
}
