<?php

namespace App\News\Service;

use App\News\Dto\NewsDetailElementDto;
use App\News\Dto\NewsListDto;
use App\News\Dto\NewsListingElementDto;
use App\News\DtoFactory\NewsCategoryDtoFactory;
use App\News\DtoFactory\NewsDetailElementDtoFactory;
use App\News\DtoFactory\NewsFilterVariantsDtoFactory;
use App\News\DtoFactory\NewsListDtoFactory;
use App\News\DtoFactory\NewsListingElementDtoFactory;
use App\News\EntityFabric\NewsEntityFabric;
use App\News\Request\NewsCreateRequest;
use App\News\Request\NewsDetailRequest;
use App\News\Request\NewsFullUpdateRequest;
use App\News\Request\NewsListingRequest;
use App\News\Request\NewsPartUpdateRequest;
use App\Shared\DtoFactory\PaginationDtoFactory;
use App\Shared\Error\NotFoundError;
use App\Shared\Repository\NewsCategoriesRepository;
use App\Shared\Repository\NewsRepository;

class NewsPrepareResponseService
{
    public function __construct(
        private readonly NewsRepository $news,
        private readonly NewsCategoriesRepository $newsCategories,
        private readonly PaginationDtoFactory $paginationFactory,
        private readonly NewsListingElementDtoFactory $listFactory,
        private readonly NewsFilterVariantsDtoFactory $filterFactory,
        private readonly NewsListDtoFactory $listingFactory,
        private readonly NewsCategoryDtoFactory $categoryFactory,
        private readonly NewsDetailElementDtoFactory $detailElementFactory,
        private readonly NewsEntityFabric $entityFabric,
    ) {
    }

    public
    function bornListDto(
        NewsListingRequest $request
    ): NewsListDto {
        $countOfNews = $this->news->getCountWithFilters(
            $request->news_category
        );

        $pagination = $this->paginationFactory->create(
            $request->page,
            $request->limit,
            $countOfNews
        );

        $offset = min(
                $pagination->pageSize,
                $countOfNews
            ) * ($pagination->currentPage - 1);

        $list = $this->listFactory->createCollection(
            $this->news->getWithFilters(
                $request->news_category,
                $request->limit,
                $offset
            )
        );

        $filters = $this->filterFactory->create(
            $this->categoryFactory->createCollection(
                $this->newsCategories->getAll()
            )
        );

        return $this->listingFactory->create(
            $pagination,
            $list,
            $filters
        );
    }

    public
    function bornMainNews(): NewsListingElementDto
    {
        return $this->listFactory->create($this->news->getMainNews());
    }

    public
    function bornDetailMainNews(): NewsDetailElementDto
    {
        return $this->detailElementFactory->create($this->news->getMainNews());
    }

    public
    function bornDetail(
        NewsDetailRequest $request
    ): NewsDetailElementDto {
        $news = $this->news->find($request->detailId);

        if ($news === null) {
            throw new NotFoundError('News not found');
        }

        return $this->detailElementFactory->create($news);
    }

    public
    function unbirth(
        NewsDetailRequest $request
    ) {
        $news = $this->news->find($request->detailId);
        if ($news === null) {
            throw new NotFoundError('News not found');
        }

        $this->news->remove($news);
    }

    public function rebirth(NewsPartUpdateRequest $request):void
    {
        $news = $this->entityFabric->hardUpdate($request);
        $this->news->createUpdate($news);
    }

    public function reborn(NewsFullUpdateRequest $request): void
    {
        $news = $this->entityFabric->softUpdate($request);
        $this->news->createUpdate($news);
    }

    public function birth(NewsCreateRequest $request): void
    {
        $news = $this->entityFabric->create($request);
        $this->news->createUpdate($news);
    }
}
