<?php

namespace App\Tests\RestaurantTests\Controller;

use App\Restaurants\Collection\KitchenCollection;
use App\Restaurants\Collection\RestaurantTypeCollection;
use App\Restaurants\Collection\TagCollection;
use App\Restaurants\Dto\KitchenTypeDto;
use App\Restaurants\Dto\RestaurantDetailElementDto;
use App\Restaurants\Dto\RestaurantFilterVariantsDto;
use App\Restaurants\Dto\RestaurantListingElementDto;
use App\Restaurants\Dto\RestaurantTypeDto;
use App\Restaurants\Dto\TagDto;
use App\Shared\Collection\FileCollection;
use App\Shared\Collection\ListingCollection;
use App\Shared\Collection\StringCollection;
use App\Shared\Dto\FileDto;
use App\Shared\Dto\PaginationDto;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\HttpFoundation\JsonResponse;

class RestaurantsControllerTests extends WebTestCase
{
    // Тест restaurant
    public function testRightUuid(): void
    {
        $client = static::createClient();
        $client->request(
            'GET',
            '/api/v1/restaurants/00000000-0000-0000-0000-000000000000'
        );
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();

        self::assertSame(
            (new JsonResponse(
                new RestaurantDetailElementDto(
                    '00000000-0000-0000-0000-000000000000',
                    'Restaurant1',
                    'restaurant_1',
                    '0.0.0.0',
                    new RestaurantTypeDto(
                        '00000000-0000-0000-0000-000000000001',
                        'Type2',
                        'type_2',
                    ),
                    'check1',
                    'check1Info',
                    new KitchenCollection(
                        [
                            new KitchenTypeDto(
                                '00000000-0000-0000-0000-000000000000',
                                'Kitchen1',
                                'kitchen_1'
                            ),
                        ]
                    ),
                    new StringCollection(['phone1' => '+7000000000']),
                    new StringCollection(['email' => 'base@gmail.com']),
                    new StringCollection(['address' => 'somewhere']),
                    new TagCollection(
                        [
                            new TagDto(
                                'setTags1',
                                new StringCollection(['tag1', 'tag2', 'tag3'])
                            ),
                            new TagDto(
                                'setTags2',
                                new StringCollection(['tag1', 'tag2', 'tag3'])
                            ),
                        ]
                    ),
                    'www.restaurant.ru',
                    new FileDto(
                        '00000000-0000-0000-0000-000000000001',
                        'image1',
                        'description',
                        512,
                        'jpg',
                        '/here/'
                    ),
                    new FileCollection([])
                )
            ))->getContent(),
            $data,
            'Полученная детальная новость инккоректена'
        );
    }

    public function testNonExistedUuid(): void
    {
        $client = static::createClient();
        $client->request(
            'GET',
            '/api/v1/restaurants/00000000-0000-0000-0000-000000000005'
        );

        self::assertResponseStatusCodeSame(404);
    }

    public function testCorruptUuid(): void
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/restaurants/net-a-uuid');

        self::assertResponseStatusCodeSame(400);
    }

    //Тест restaurants
    public function testNullRequest(): void
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/restaurants/');
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();

        /** Проверка пагинации */
        self::assertStringContainsString(
            (new JsonResponse(new PaginationDto(1, 1, 12),))->getContent(),
            $data,
            'Пагинация инкорректна при пустом реквесте'
        );

        /** Проверка получемых новостей */
        self::assertStringContainsString(
            (new JsonResponse(
                new ListingCollection(
                    [
                        new RestaurantListingElementDto(
                            '00000000-0000-0000-0000-000000000000',
                            'Restaurant1',
                            'restaurant_1',
                            new RestaurantTypeDto(
                                '00000000-0000-0000-0000-000000000001',
                                'Type2',
                                'type_2',
                            ),
                            'api/v1/restaurants/00000000-0000-0000-0000-000000000000',
                            'check1',
                            new FileDto(
                                '00000000-0000-0000-0000-000000000001',
                                'image1',
                                'description',
                                512,
                                'jpg',
                                '/here/'
                            ),
                        ),
                    ]
                )
            ))->getContent(),
            $data,
            'Полученный список новостей инккоректен при пустом реквесте'
        );

        /** Проверка фильтров */
        self::assertStringContainsString(
            (new JsonResponse(
                new RestaurantFilterVariantsDto(
                    new RestaurantTypeCollection(
                        [
                            new RestaurantTypeDto(
                                '00000000-0000-0000-0000-000000000000',
                                'Type1',
                                'type_1',
                            ),
                            new RestaurantTypeDto(
                                '00000000-0000-0000-0000-000000000001',
                                'Type2',
                                'type_2',
                            ),
                        ]
                    ),
                    new KitchenCollection(
                        [
                            new KitchenTypeDto(
                                '00000000-0000-0000-0000-000000000000',
                                'Kitchen1',
                                'kitchen_1',
                            )
                        ]
                    )
                )
            ))->getContent(),
            $data,
            'Пагинация инкорректна при пустом реквесте'
        );
    }

    public function testNegativePagination(): void
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/restaurants/?page=-10&limit=-10');
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();

        /** Проверка пагинации */
        self::assertStringContainsString(
            (new JsonResponse(new PaginationDto(1, 1, 12),))->getContent(),
            $data,
            'Пагинация инкорректна при негативных значениях'
        );
    }

    public function testPageOverListCountPagination(): void
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/restaurants/?page=14');
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();

        /** Проверка пагинации */
        self::assertStringContainsString(
            (new JsonResponse(new PaginationDto(1, 1, 12),))->getContent(),
            $data,
            'Пагинация инкорректна при позитивных значениях'
        );
    }

    public function testRightFilter(): void
    {
        $client = static::createClient();
        $client->request(
            'GET',
            '/api/v1/news/?restaurant_type_id=00000000-0000-0000-0000-000000000000'
        );

        self::assertResponseIsSuccessful();
    }

    public function testCorruptFilter(): void
    {
        $client = static::createClient();
        $client->request(
            'GET',
            '/api/v1/news/?restaurant_type_id=looks-like-not-a-UUId'
        );

        self::assertResponseStatusCodeSame(200);
    }
}