<?php

namespace App\Restaurants\Service;

use App\Restaurants\Dto\RestaurantDetailElementDto;
use App\Restaurants\Dto\RestaurantListDto;
use App\Restaurants\DtoFactory\KitchenTypeDtoFactory;
use App\Restaurants\DtoFactory\RestaurantDetailElementDtoFactory;
use App\Restaurants\DtoFactory\RestaurantFilterVariantsDtoFactory;
use App\Restaurants\DtoFactory\RestaurantListDtoFactory;
use App\Restaurants\DtoFactory\RestaurantListingElementDtoFactory;
use App\Restaurants\DtoFactory\RestaurantTypeDtoFactory;
use App\Restaurants\Request\RestaurantCreateRequest;
use App\Restaurants\Request\RestaurantDetailRequest;
use App\Restaurants\Request\RestaurantFullUpdateRequest;
use App\Restaurants\Request\RestaurantListingRequest;
use App\Restaurants\Request\RestaurantPartUpdateRequest;
use App\Shared\DtoFactory\PaginationDtoFactory;
use App\Shared\EntityFactory\RestaurantEntityFactory;
use App\Shared\Error\NotFoundError;
use App\Shared\Repository\KitchensRepository;
use App\Shared\Repository\RestaurantsRepository;
use App\Shared\Repository\RestaurantTypesRepository;
use JsonException;

class RestaurantPrepareResponseService
{
    public function __construct(
        private readonly RestaurantsRepository $restaurants,
        private readonly RestaurantTypesRepository $types,
        private readonly KitchensRepository $kitchens,
        private readonly PaginationDtoFactory $paginationFactory,
        private readonly RestaurantListingElementDtoFactory $listFactory,
        private readonly RestaurantFilterVariantsDtoFactory $filterFactory,
        private readonly RestaurantTypeDtoFactory $typeFactory,
        private readonly KitchenTypeDtoFactory $kitchenFactory,
        private readonly RestaurantListDtoFactory $listingFactory,
        private readonly RestaurantDetailElementDtoFactory $detailFactory,
        private readonly RestaurantEntityFactory $entityFactory,
    ) {
    }

    /**
     * @param RestaurantListingRequest $request
     * @return RestaurantListDto
     */
    public function bornListDto(RestaurantListingRequest $request
    ): RestaurantListDto {
        $countOfRestaurants = $this
            ->restaurants
            ->getCountWithFilters(
                $request->restaurant_type_id,
                $request->kitchen_id,
            );

        $pagination = $this->paginationFactory->create(
            $request->page,
            $request->limit,
            $countOfRestaurants,
        );

        $offset = min(
                $pagination->pageSize,
                $countOfRestaurants
            ) * ($pagination->currentPage - 1);

        $list = $this->listFactory->createCollection(
            $this->restaurants->getWithFilters(
                $request->kitchen_id,
                $request->restaurant_type_id,
                $pagination->pageSize,
                $offset,
            ),
        );


        $filters = $this->filterFactory->create(
            $this->typeFactory->createCollection($this->types->getAll()),
            $this->kitchenFactory->createCollection($this->kitchens->getAll())
        );

        return $this->listingFactory->create(
            $pagination,
            $list,
            $filters
        );
    }

    /**
     * @param RestaurantDetailRequest $request
     * @return RestaurantDetailElementDto
     * @throws JsonException
     */
    public
    function bornDetailElement(
        RestaurantDetailRequest $request
    ): RestaurantDetailElementDto {
        $restaurant = $this->restaurants->find($request->detailId);

        if ($restaurant === null) {
            throw new NotFoundError('Restaurant not found');
        }

        return $this->detailFactory->create($restaurant);
    }

    public function unbirth(
        RestaurantDetailRequest $request
    ) {
        $restaurants = $this->restaurants->find($request->detailId);
        if ($restaurants === null) {
            throw new NotFoundError('Restaurant not found');
        }

        $this->restaurants->remove($restaurants);
    }

    public function rebirth(RestaurantPartUpdateRequest $request): void
    {
        $news = $this->entityFactory->hardUpdate($request);
        $this->restaurants->createUpdate($news);
    }

    public function reborn(RestaurantFullUpdateRequest $request): void
    {
        $news = $this->entityFactory->softUpdate($request);
        $this->restaurants->createUpdate($news);
    }

    public function birth(RestaurantCreateRequest $request): void
    {
        $news = $this->entityFactory->create($request);
        $this->restaurants->createUpdate($news);
    }
}
