<?php

namespace App\Tests\NewsTests\Controller;

use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

class NewsControllerTests extends WebTestCase
{
    // Тест oneNews
    public function testRightUuid(): void
    {
        $client = static::createClient();
        $client->request(
            'GET',
            '/api/v1/news/00000000-0000-0000-0000-000000000000'
        );
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();

        /** Проверка получемых новостей */
        self::assertStringContainsString(
            '"id":"00000000-0000-0000-0000-000000000000","name":"news1","description":"Preview text","createAt":"17.05.2024","text":"Detail text","image":{"id":"00000000-0000-0000-0000-000000000000","name":"file1","description":"fileDescription","size":1024,"type":"png","url":"\/somewhere\/"',
            $data,
            'Полученная детальная новость инккоректена'
        );
    }

    public function testNonExistedUuid(): void
    {
        $client = static::createClient();
        $client->request(
            'GET',
            '/api/v1/news/00000000-0000-0000-0000-000000000005'
        );
        $data = $client->getResponse()->getContent();

        self::assertResponseStatusCodeSame(404);
    }

    public function testCorruptUuid(): void
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/news/looks-like-not-a-UUId');

        self::assertResponseStatusCodeSame(400);
    }


    // Тест news
    public function testNullRequest(): void
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/news/');
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();

        /** Проверка пагинации */
        self::assertStringContainsString(
            '"pagination":{"currentPage":1,"pages":1,"pageSize":12}',
            $data,
            'Пагинация инкорректна при пустом реквесте'
        );

        /** Проверка получемых новостей */
        self::assertStringContainsString(
            '"list":[{"id":"00000000-0000-0000-0000-000000000000","name":"news1","detailLink":"api\/v1\/news\/00000000-0000-0000-0000-000000000000","createAt":"17.05.2024","description":"Preview text","image":{"id":"00000000-0000-0000-0000-000000000000","name":"file1","description":"fileDescription","size":1024,"type":"png","url":"\/somewhere\/"}},{"id":"00000000-0000-0000-0000-000000000001","name":"news2","detailLink":"api\/v1\/news\/00000000-0000-0000-0000-000000000001","createAt":"17.05.2024","description":null,"image":null}]',
            $data,
            'Полученный список новостей инккоректен при пустом реквесте'
        );

        /** Проверка фильтров */
        self::assertStringContainsString(
            '"filterVariants"' .
            ':{"category":[{"id":"00000000-0000-0000-0000-000000000000",' .
            '"name":"newsCategory1","code":"newsCategory_1"},' .
            '{"id":"00000000-0000-0000-0000-000000000001",' .
            '"name":"newsCategory2","code":"newsCategory_2"}]}',
            $data,
            'Пагинация инкорректна при пустом реквесте'
        );
    }

    public function testNegativePagination(): void
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/news/?page=-10&limit=-10');
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();

        /** Проверка пагинации */
        self::assertStringContainsString(
            '"pagination":{"currentPage":1,"pages":1,"pageSize":12}',
            $data,
            'Пагинация инкорректна при негативных значениях'
        );
    }

    public function testPageOverListCountPagination(): void
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/news/?page=14');
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();

        /** Проверка пагинации */
        self::assertStringContainsString(
            '"pagination":{"currentPage":1,"pages":1,"pageSize":12}',
            $data,
            'Пагинация инкорректна при позитивных значениях'
        );
    }

    public function testRightFilter(): void
    {
        $client = static::createClient();
        $client->request(
            'GET',
            '/api/v1/news/?news_category=00000000-0000-0000-0000-000000000000'
        );
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();

        /** Проверка получемых новостей */
        self::assertStringNotContainsString(
            '"list":[{"id":"00000000-0000-0000-0000-000000000000",' .
            '"name":"news1","createAt":"2024-05-15 09:55:45",' .
            '"detailLink":"api\/v1\/news\/00000000-0000-0000-0000-000000000000"' .
            ',"description":"Preview text",' .
            '"image":{"id":"00000000-0000-0000-0000-000000000000",' .
            '"name":"file1","description":"fileDescription","size":1024,' .
            '"type":"png","url":"\/somewhere\/"}},' .
            '{"id":"00000000-0000-0000-0000-000000000001",' .
            '"name":"news2","createAt":"2024-05-15 09:55:45",' .
            '"detailLink":"api\/v1\/news\/00000000-0000-0000-0000-000000000001",' .
            '"description":null,"image":null}]',
            $data,
            'Полученный список новостей инккоректен при правильном фильтре'
        );
    }

    public function testCorruptFilter(): void
    {
        $client = static::createClient();
        $client->request(
            'GET',
            '/api/v1/news/?news_category=looks-like-not-a-UUId'
        );
        $data = $client->getResponse()->getContent();

        self::assertResponseStatusCodeSame(200);
    }


    // mainNews
    public function testMainNews()
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/news/mainNews');
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();
    }

    public function testSearch()
    {
        $client = static::createClient();
        $client->request('GET', '/api/v1/news/search');
        $data = $client->getResponse()->getContent();

        self::assertResponseIsSuccessful();
    }
}
