<?php

namespace App\Restaurants\Service;

use App\Restaurants\DtoFactory\KitchenTypeDtoFactory;
use App\Restaurants\DtoFactory\RestaurantDetailElementDtoFactory;
use App\Restaurants\DtoFactory\RestaurantFilterVariantsDtoFactory;
use App\Restaurants\DtoFactory\RestaurantListDtoFactory;
use App\Restaurants\DtoFactory\RestaurantListingElementDtoFactory;
use App\Restaurants\DtoFactory\RestaurantTypeDtoFactory;
use App\Shared\Abstraction\AbstractDetailElement;
use App\Shared\Abstraction\AbstractListDto;
use App\Shared\Abstraction\ServiceInterface;
use App\Shared\DtoFactory\PaginationDtoFactory;
use App\Shared\Error\NonValidUuidError;
use App\Shared\Error\NotFoundError;
use App\Shared\Repository\KitchensRepository;
use App\Shared\Repository\RestaurantsRepository;
use App\Shared\Repository\RestaurantTypesRepository;
use App\Shared\Service\ValidateService;
use JsonException;

class RestaurantPrepareRequestService implements ServiceInterface
{
    public function __construct(
        private readonly RestaurantsRepository $restaurants,
        private readonly RestaurantTypesRepository $types,
        private readonly KitchensRepository $kitchens,
        private readonly PaginationDtoFactory $paginationFactory,
        private readonly RestaurantListingElementDtoFactory $listFactory,
        private readonly RestaurantFilterVariantsDtoFactory $filterFactory,
        private readonly RestaurantTypeDtoFactory $typeFactory,
        private readonly KitchenTypeDtoFactory $kitchenFactory,
        private readonly RestaurantListDtoFactory $listingFactory,
        private readonly RestaurantDetailElementDtoFactory $detailFactory,
        private readonly ValidateService $validate,
    ) {
    }

    /**
     * @param int $page
     * @param int $limit
     * @param array|null $filters
     * @return AbstractListDto
     */
    public function bornListDto(
        int $page,
        int $limit,
        ?array $filters
    ): AbstractListDto {
        $filters['restaurant_type_id'] = $filters[0];
        $filters['kitchen_id'] = $filters[1];
        if (!$this->validate->isValidUuid($filters['restaurant_type_id'])) {
            $filters['news_category'] = null;
        }
        if (!$this->validate->isValidUuid($filters['kitchen_id'])) {
            $filters['news_category'] = null;
        }
        $this->validate->correctPagination($page, $limit);

        $countOfRestaurants = $this
            ->restaurants
            ->getCountWithFilters(
                $filters['restaurant_type_id'],
                $filters['kitchen_id']
            );

        $pagination = $this->paginationFactory->create(
            $page,
            $limit,
            $countOfRestaurants,
        );

        $offset = min(
                $pagination->pageSize,
                $countOfRestaurants
            ) * ($pagination->currentPage - 1);

        $list = $this->listFactory->createCollection(
            $this->restaurants->getWithFilters(
                $filters['kitchen_id'],
                $filters['restaurant_type_id'],
                $pagination->pageSize,
                $offset,
            ),
        );


        $filters = $this->filterFactory->create(
            $this->typeFactory->createCollection($this->types->getAll()),
            $this->kitchenFactory->createCollection($this->kitchens->getAll())
        );

        return $this->listingFactory->create(
            $pagination,
            $list,
            $filters
        );
    }

    /**
     * @param string $detailId
     * @return AbstractDetailElement
     * @throws JsonException
     */
    public
    function bornDetailElement(
        string $detailId
    ): AbstractDetailElement {
        if (!$this->validate->isValidUuid($detailId)) {
            throw new NonValidUuidError();
        }

        $restaurant = $this->restaurants->find($detailId);

        if ($restaurant === null) {
            throw new NotFoundError('Restaurant not found');
        }

        return $this->detailFactory->create($restaurant);
    }
}
